<?php

namespace AutoForumSyncPro\Tests;

use AutoForumSyncPro\Service\ContentFetcher;
use AutoForumSyncPro\Service\SpamFilter;
use AutoForumSyncPro\Service\KeywordReplacer;
use AutoForumSyncPro\Entity\TargetSite;

class ContentFetcherTest
{
    protected $contentFetcher;
    protected $targetSite;

    public function __construct()
    {
        $this->contentFetcher = new ContentFetcher();
        $this->setupTargetSite();
    }

    /**
     * Test hedef siteyi ayarla
     */
    protected function setupTargetSite()
    {
        $siteData = [
            'target_id' => 1,
            'site_name' => 'Test Site',
            'base_url' => 'https://example.com',
            'site_type' => 'xenforo',
            'enabled' => 1,
            'default_user_id' => 1,
            'default_forum_id' => 1,
            'delay_between_requests' => 1
        ];
        
        $this->targetSite = new TargetSite($siteData);
    }

    /**
     * URL geçerlilik testi
     */
    public function testValidUrl()
    {
        $validUrls = [
            'https://example.com/threads/test.123/',
            'http://example.com/forums/test-thread.456/',
            'https://subdomain.example.com/threads/another-test.789/'
        ];

        foreach ($validUrls as $url) {
            $result = $this->contentFetcher->fetchFromUrl($url);
            $this->assertNotNull($result, "URL geçerli olmalı: {$url}");
        }
    }

    /**
     * Geçersiz URL testi
     */
    public function testInvalidUrl()
    {
        $invalidUrls = [
            'not-a-url',
            'ftp://example.com',
            'javascript:alert(1)',
            ''
        ];

        foreach ($invalidUrls as $url) {
            $result = $this->contentFetcher->fetchFromUrl($url);
            $this->assertNull($result, "URL geçersiz olmalı: {$url}");
        }
    }

    /**
     * Bağlantı testi
     */
    public function testConnection()
    {
        $testUrl = 'https://httpbin.org/status/200';
        $result = $this->contentFetcher->testConnection($testUrl);
        $this->assertTrue($result, 'Bağlantı testi başarısız');
    }

    /**
     * Spam filtreleme testi
     */
    public function testSpamFiltering()
    {
        $spamContent = [
            'title' => 'Buy cheap viagra now!',
            'content' => 'This is spam content with bad words',
            'tags' => ['spam', 'viagra']
        ];

        $spamFilter = new SpamFilter();
        $isSpam = $spamFilter->isSpam($spamContent);
        $this->assertTrue($isSpam, 'Spam içerik tespit edilmeli');
    }

    /**
     * Anahtar kelime değiştirme testi
     */
    public function testKeywordReplacement()
    {
        $content = [
            'title' => 'Original Site Title',
            'content' => 'This content mentions original-site.com',
            'tags' => ['original', 'site']
        ];

        $keywordReplacer = new KeywordReplacer();
        $keywordReplacer->addReplacement('original-site.com', 'my-site.com');
        $keywordReplacer->addReplacement('Original Site', 'My Site');

        $result = $keywordReplacer->replace($content);
        
        $this->assertStringContains('my-site.com', $result['content']);
        $this->assertStringContains('My Site', $result['title']);
    }

    /**
     * Duplicate kontrol testi
     */
    public function testDuplicateDetection()
    {
        $content1 = [
            'unique_id' => 'test_123',
            'title' => 'Test Title',
            'content' => 'Test content'
        ];

        $content2 = [
            'unique_id' => 'test_123',
            'title' => 'Test Title 2',
            'content' => 'Test content 2'
        ];

        // İlk içeriği kaydet
        $this->contentFetcher->setTargetSite($this->targetSite);
        $result1 = $this->contentFetcher->fetchFromUrl('https://example.com/test1');
        
        // Aynı unique_id ile ikinci içerik
        $result2 = $this->contentFetcher->fetchFromUrl('https://example.com/test2');
        
        $this->assertNull($result2, 'Duplicate içerik tespit edilmeli');
    }

    /**
     * Toplu içerik çekme testi
     */
    public function testMultipleContentFetching()
    {
        $urls = [
            'https://example.com/thread1',
            'https://example.com/thread2',
            'https://example.com/thread3'
        ];

        $this->contentFetcher->setTargetSite($this->targetSite);
        $results = $this->contentFetcher->fetchMultiple($urls);
        
        $this->assertIsArray($results);
        $this->assertGreaterThan(0, count($results));
    }

    /**
     * Parser testi
     */
    public function testParser()
    {
        $html = '
        <html>
            <head><title>Test Thread Title</title></head>
            <body>
                <h1 class="p-title-value">Test Thread Title</h1>
                <div class="bbWrapper">
                    <p>This is test content with <strong>bold text</strong>.</p>
                    <img src="/images/test.jpg" alt="Test image">
                </div>
                <div class="tagList">
                    <span class="tag">test</span>
                    <span class="tag">example</span>
                </div>
            </body>
        </html>';

        $this->contentFetcher->setTargetSite($this->targetSite);
        $result = $this->contentFetcher->fetchFromUrl('https://example.com/threads/test.123/');
        
        $this->assertNotNull($result);
        $this->assertEquals('Test Thread Title', $result['title']);
        $this->assertStringContains('test content', $result['content']);
        $this->assertContains('test', $result['tags']);
    }

    /**
     * Test sonuçlarını raporla
     */
    public function runTests()
    {
        $tests = [
            'testValidUrl',
            'testInvalidUrl', 
            'testConnection',
            'testSpamFiltering',
            'testKeywordReplacement',
            'testDuplicateDetection',
            'testMultipleContentFetching',
            'testParser'
        ];

        $passed = 0;
        $failed = 0;

        foreach ($tests as $test) {
            try {
                $this->$test();
                echo "✓ {$test} - PASSED\n";
                $passed++;
            } catch (\Exception $e) {
                echo "✗ {$test} - FAILED: " . $e->getMessage() . "\n";
                $failed++;
            }
        }

        echo "\nTest Sonuçları:\n";
        echo "Geçen: {$passed}\n";
        echo "Başarısız: {$failed}\n";
        echo "Toplam: " . ($passed + $failed) . "\n";
    }

    /**
     * Assert helper
     */
    protected function assertNotNull($value, $message = '')
    {
        if ($value === null) {
            throw new \Exception($message ?: 'Değer null olmamalı');
        }
    }

    protected function assertNull($value, $message = '')
    {
        if ($value !== null) {
            throw new \Exception($message ?: 'Değer null olmalı');
        }
    }

    protected function assertTrue($value, $message = '')
    {
        if (!$value) {
            throw new \Exception($message ?: 'Değer true olmalı');
        }
    }

    protected function assertStringContains($needle, $haystack, $message = '')
    {
        if (strpos($haystack, $needle) === false) {
            throw new \Exception($message ?: "String '{$needle}' içermeli");
        }
    }

    protected function assertContains($needle, $haystack, $message = '')
    {
        if (!in_array($needle, $haystack)) {
            throw new \Exception($message ?: "Array '{$needle}' içermeli");
        }
    }

    protected function assertIsArray($value, $message = '')
    {
        if (!is_array($value)) {
            throw new \Exception($message ?: 'Değer array olmalı');
        }
    }

    protected function assertGreaterThan($expected, $actual, $message = '')
    {
        if ($actual <= $expected) {
            throw new \Exception($message ?: "Değer {$expected}'den büyük olmalı");
        }
    }
}
