<?php

namespace AutoForumSyncPro;

class Setup
{
    public static function install()
    {
        // Veritabanı tablolarını oluştur
        self::createTables();
        
        // Varsayılan verileri ekle
        self::insertDefaultData();
        
        // İzinleri ayarla
        self::setupPermissions();
    }

    public static function uninstall()
    {
        // Veritabanı tablolarını sil
        self::dropTables();
        
        // Eklenti ayarlarını sil
        self::removeOptions();
    }

    protected static function createTables()
    {
        $db = \XF::db();
        
        // Hedef siteler tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_targets` (
                `target_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `site_name` varchar(255) NOT NULL,
                `base_url` varchar(500) NOT NULL,
                `site_type` enum('xenforo','vbulletin','phpbb','custom') NOT NULL,
                `enabled` tinyint(1) NOT NULL DEFAULT '1',
                `default_user_id` int(10) unsigned NOT NULL,
                `default_forum_id` int(10) unsigned NOT NULL,
                `delay_between_requests` int(10) unsigned NOT NULL DEFAULT '1',
                `last_run_time` int(10) unsigned DEFAULT NULL,
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`target_id`),
                KEY `enabled` (`enabled`),
                KEY `site_type` (`site_type`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // İçerik logları tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_content` (
                `content_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `unique_id` varchar(255) NOT NULL,
                `source_url` varchar(500) NOT NULL,
                `thread_id` int(10) unsigned DEFAULT NULL,
                `post_id` int(10) unsigned DEFAULT NULL,
                `user_id` int(10) unsigned NOT NULL,
                `status` enum('published','pending','failed') NOT NULL DEFAULT 'published',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`content_id`),
                UNIQUE KEY `unique_id` (`unique_id`),
                KEY `status` (`status`),
                KEY `user_id` (`user_id`),
                KEY `created_date` (`created_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Spam filtreleri tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_blacklist` (
                `blacklist_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `word` varchar(255) NOT NULL,
                `type` enum('word','regex') NOT NULL DEFAULT 'word',
                `enabled` tinyint(1) NOT NULL DEFAULT '1',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`blacklist_id`),
                KEY `enabled` (`enabled`),
                KEY `type` (`type`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Beyaz liste tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_whitelist` (
                `whitelist_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `word` varchar(255) NOT NULL,
                `enabled` tinyint(1) NOT NULL DEFAULT '1',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`whitelist_id`),
                KEY `enabled` (`enabled`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Anahtar kelime değiştirme tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_replacements` (
                `replacement_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `search_text` varchar(500) NOT NULL,
                `replace_text` varchar(500) NOT NULL,
                `type` enum('simple','regex','url') NOT NULL DEFAULT 'simple',
                `enabled` tinyint(1) NOT NULL DEFAULT '1',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`replacement_id`),
                KEY `enabled` (`enabled`),
                KEY `type` (`type`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // HTML filtreleri tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_html_filters` (
                `filter_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `element` varchar(100) NOT NULL,
                `enabled` tinyint(1) NOT NULL DEFAULT '1',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`filter_id`),
                KEY `enabled` (`enabled`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Bekleyen içerikler tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_pending` (
                `pending_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `unique_id` varchar(255) NOT NULL,
                `source_url` varchar(500) NOT NULL,
                `title` varchar(500) NOT NULL,
                `content` longtext NOT NULL,
                `tags` text,
                `user_id` int(10) unsigned NOT NULL,
                `forum_id` int(10) unsigned NOT NULL,
                `status` enum('pending_approval','approved','rejected') NOT NULL DEFAULT 'pending_approval',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`pending_id`),
                UNIQUE KEY `unique_id` (`unique_id`),
                KEY `status` (`status`),
                KEY `user_id` (`user_id`),
                KEY `forum_id` (`forum_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Zamanlanmış çalışmalar tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_schedules` (
                `schedule_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `target_id` int(10) unsigned NOT NULL,
                `interval_minutes` int(10) unsigned NOT NULL,
                `next_run` int(10) unsigned NOT NULL,
                `enabled` tinyint(1) NOT NULL DEFAULT '1',
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`schedule_id`),
                KEY `target_id` (`target_id`),
                KEY `next_run` (`next_run`),
                KEY `enabled` (`enabled`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Log tablosu
        $db->query("
            CREATE TABLE IF NOT EXISTS `xf_auto_forum_sync_logs` (
                `log_id` int(10) unsigned NOT NULL AUTO_INCREMENT,
                `timestamp` varchar(50) NOT NULL,
                `level` enum('debug','info','warning','error') NOT NULL,
                `message` text NOT NULL,
                `context` text,
                `created_date` int(10) unsigned NOT NULL,
                PRIMARY KEY (`log_id`),
                KEY `level` (`level`),
                KEY `created_date` (`created_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }

    protected static function insertDefaultData()
    {
        $db = \XF::db();
        $time = \XF::$time;

        // Varsayılan spam filtreleri
        $blacklistWords = [
            'viagra', 'casino', 'porn', 'spam', 'pills', 'casino', 'gambling'
        ];

        foreach ($blacklistWords as $word) {
            $db->insert('xf_auto_forum_sync_blacklist', [
                'word' => $word,
                'type' => 'word',
                'enabled' => 1,
                'created_date' => $time
            ]);
        }

        // Varsayılan HTML filtreleri
        $htmlElements = [
            'script', 'style', 'iframe', 'object', 'embed'
        ];

        foreach ($htmlElements as $element) {
            $db->insert('xf_auto_forum_sync_html_filters', [
                'element' => $element,
                'enabled' => 1,
                'created_date' => $time
            ]);
        }

        // Varsayılan anahtar kelime değiştirmeleri
        $replacements = [
            ['search_text' => 'eski-site.com', 'replace_text' => 'yeni-site.com', 'type' => 'url'],
            ['search_text' => 'Eski Site', 'replace_text' => 'Yeni Site', 'type' => 'simple']
        ];

        foreach ($replacements as $replacement) {
            $db->insert('xf_auto_forum_sync_replacements', [
                'search_text' => $replacement['search_text'],
                'replace_text' => $replacement['replace_text'],
                'type' => $replacement['type'],
                'enabled' => 1,
                'created_date' => $time
            ]);
        }
    }

    protected static function setupPermissions()
    {
        // Admin izinlerini ayarla - XenForo otomatik olarak XML'den yükler
        // Bu metod boş bırakılabilir
    }

    protected static function dropTables()
    {
        $db = \XF::db();
        
        $tables = [
            'xf_auto_forum_sync_logs',
            'xf_auto_forum_sync_schedules',
            'xf_auto_forum_sync_pending',
            'xf_auto_forum_sync_html_filters',
            'xf_auto_forum_sync_replacements',
            'xf_auto_forum_sync_whitelist',
            'xf_auto_forum_sync_blacklist',
            'xf_auto_forum_sync_content',
            'xf_auto_forum_sync_targets'
        ];

        foreach ($tables as $table) {
            $db->query("DROP TABLE IF EXISTS `{$table}`");
        }
    }

    protected static function removeOptions()
    {
        $db = \XF::db();
        
        $options = [
            'autoForumSyncEnabled',
            'autoForumSyncDefaultUser',
            'autoForumSyncDefaultForum',
            'autoForumSyncMaxPostsPerRun',
            'autoForumSyncDelayBetweenPosts'
        ];

        foreach ($options as $option) {
            $db->delete('xf_option', 'option_id = ?', $option);
        }
    }
}
