<?php

namespace AutoForumSyncPro\Service;

class Logger
{
    protected $logLevel;
    protected $logToDatabase;
    protected $logToFile;

    public function __construct()
    {
        $this->logLevel = \XF::options()->autoForumSyncLogLevel ?? 'info';
        $this->logToDatabase = true;
        $this->logToFile = true;
    }

    /**
     * Info log
     */
    public function info($message, $context = [])
    {
        $this->log('info', $message, $context);
    }

    /**
     * Warning log
     */
    public function warning($message, $context = [])
    {
        $this->log('warning', $message, $context);
    }

    /**
     * Error log
     */
    public function error($message, $context = [])
    {
        $this->log('error', $message, $context);
    }

    /**
     * Debug log
     */
    public function debug($message, $context = [])
    {
        $this->log('debug', $message, $context);
    }

    /**
     * Log yaz
     */
    protected function log($level, $message, $context = [])
    {
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = [
            'timestamp' => $timestamp,
            'level' => $level,
            'message' => $message,
            'context' => json_encode($context)
        ];

        // Veritabanına log
        if ($this->logToDatabase) {
            $this->logToDatabase($logEntry);
        }

        // Dosyaya log
        if ($this->logToFile) {
            $this->logToFile($logEntry);
        }

        // XenForo error log
        if ($level === 'error') {
            \XF::logError("AutoForumSync: " . $message);
        }
    }

    /**
     * Veritabanına log yaz
     */
    protected function logToDatabase($logEntry)
    {
        try {
            \XF::db()->insert('xf_auto_forum_sync_logs', [
                'timestamp' => $logEntry['timestamp'],
                'level' => $logEntry['level'],
                'message' => $logEntry['message'],
                'context' => $logEntry['context'],
                'created_date' => \XF::$time
            ]);
        } catch (\Exception $e) {
            // Log yazma hatası, sessizce geç
        }
    }

    /**
     * Dosyaya log yaz
     */
    protected function logToFile($logEntry)
    {
        try {
            $logFile = \XF::getRootDirectory() . '/internal_data/auto_forum_sync.log';
            $logLine = "[{$logEntry['timestamp']}] {$logEntry['level']}: {$logEntry['message']}\n";
            
            file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
        } catch (\Exception $e) {
            // Log yazma hatası, sessizce geç
        }
    }

    /**
     * Log seviyesini kontrol et
     */
    protected function shouldLog($level)
    {
        $levels = ['debug' => 0, 'info' => 1, 'warning' => 2, 'error' => 3];
        $currentLevel = isset($levels[$this->logLevel]) ? $levels[$this->logLevel] : 1;
        $messageLevel = isset($levels[$level]) ? $levels[$level] : 1;
        
        return $messageLevel >= $currentLevel;
    }

    /**
     * Logları temizle
     */
    public function cleanLogs($days = 30)
    {
        $cutoffTime = \XF::$time - ($days * 24 * 60 * 60);
        
        // Veritabanı loglarını temizle
        \XF::db()->delete(
            'xf_auto_forum_sync_logs',
            'created_date < ?',
            $cutoffTime
        );

        // Dosya loglarını temizle
        $logFile = \XF::getRootDirectory() . '/internal_data/auto_forum_sync.log';
        if (file_exists($logFile)) {
            $lines = file($logFile);
            $cutoffDate = date('Y-m-d H:i:s', $cutoffTime);
            
            $filteredLines = array_filter($lines, function($line) use ($cutoffDate) {
                preg_match('/^\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\]/', $line, $matches);
                return isset($matches[1]) && $matches[1] >= $cutoffDate;
            });
            
            file_put_contents($logFile, implode('', $filteredLines));
        }
    }

    /**
     * Log istatistikleri
     */
    public function getLogStats()
    {
        $stats = \XF::db()->fetchRow(
            "SELECT 
                COUNT(*) as total_logs,
                SUM(CASE WHEN level = 'error' THEN 1 ELSE 0 END) as error_count,
                SUM(CASE WHEN level = 'warning' THEN 1 ELSE 0 END) as warning_count,
                SUM(CASE WHEN level = 'info' THEN 1 ELSE 0 END) as info_count,
                SUM(CASE WHEN level = 'debug' THEN 1 ELSE 0 END) as debug_count
             FROM xf_auto_forum_sync_logs 
             WHERE created_date >= ?",
            [\XF::$time - (7 * 24 * 60 * 60)] // Son 7 gün
        );

        return $stats;
    }

    /**
     * Son logları al
     */
    public function getRecentLogs($limit = 100)
    {
        return \XF::db()->fetchAll(
            "SELECT * FROM xf_auto_forum_sync_logs 
             ORDER BY created_date DESC 
             LIMIT ?",
            [$limit]
        );
    }

    /**
     * Hata loglarını al
     */
    public function getErrorLogs($limit = 50)
    {
        return \XF::db()->fetchAll(
            "SELECT * FROM xf_auto_forum_sync_logs 
             WHERE level = 'error' 
             ORDER BY created_date DESC 
             LIMIT ?",
            [$limit]
        );
    }

    /**
     * Log seviyesini ayarla
     */
    public function setLogLevel($level)
    {
        $this->logLevel = $level;
    }

    /**
     * Log hedeflerini ayarla
     */
    public function setLogTargets($database = true, $file = true)
    {
        $this->logToDatabase = $database;
        $this->logToFile = $file;
    }
}
