<?php

namespace AutoForumSyncPro\Parser;

use Symfony\Component\DomCrawler\Crawler;

class XenforoParser extends AbstractParser
{
    /**
     * XenForo içeriği parse et
     */
    public function parse($html, $url)
    {
        $this->setCrawler($html);
        $this->setBaseUrl($url);
        
        $html = $this->cleanHtml($html);
        $this->setCrawler($html);
        
        $content = array(
            'title' => $this->extractTitle($html),
            'content' => $this->extractContent($html),
            'tags' => $this->extractTags($html),
            'unique_id' => $this->extractUniqueId($html, $url),
            'author' => $this->extractAuthor($html),
            'date' => $this->extractDate($html),
            'source_url' => $url
        );
        
        $this->validateContent($content);
        
        return $content;
    }

    /**
     * Başlık çıkar
     */
    protected function extractTitle($html)
    {
        // XenForo thread title
        $title = $this->crawler->filter('h1.p-title-value')->text();
        
        if (empty($title)) {
            // Alternatif selector
            $title = $this->crawler->filter('title')->text();
        }
        
        return trim($title);
    }

    /**
     * İçerik çıkar
     */
    protected function extractContent($html)
    {
        // XenForo post content
        $content = $this->crawler->filter('.bbWrapper')->html();
        
        if (empty($content)) {
            // Alternatif selector
            $content = $this->crawler->filter('.message-content')->html();
        }
        
        // URL'leri mutlak hale getir
        $content = $this->makeAbsoluteUrls($content, $this->baseUrl);
        
        // Resim URL'lerini optimize et
        $content = $this->optimizeImageUrls($content);
        
        return $content;
    }

    /**
     * Etiketleri çıkar
     */
    protected function extractTags($html)
    {
        $tags = array();
        
        // XenForo tags
        $this->crawler->filter('.tagList .tag')->each(function($node) use (&$tags) {
            $tags[] = $node->text();
        });
        
        return $tags;
    }

    /**
     * Benzersiz ID çıkar
     */
    protected function extractUniqueId($html, $url)
    {
        // URL'den thread ID'yi çıkar
        if (preg_match('/threads\/[^\/]+\.(\d+)/', $url, $matches)) {
            return 'xf_thread_' . $matches[1];
        }
        
        // HTML'den thread ID'yi çıkar
        if (preg_match('/data-thread-id="(\d+)"/', $html, $matches)) {
            return 'xf_thread_' . $matches[1];
        }
        
        throw new \Exception('Benzersiz ID bulunamadı');
    }

    /**
     * Yazar bilgisi çıkar
     */
    protected function extractAuthor($html)
    {
        $author = $this->crawler->filter('.message-userDetails .username')->text();
        
        if (empty($author)) {
            $author = $this->crawler->filter('.p-title-pageAction .username')->text();
        }
        
        return trim($author);
    }

    /**
     * Tarih bilgisi çıkar
     */
    protected function extractDate($html)
    {
        $dateText = $this->crawler->filter('.message-date time')->attr('datetime');
        
        if (empty($dateText)) {
            $dateText = $this->crawler->filter('.message-date time')->text();
        }
        
        return $dateText;
    }
}
