<?php

namespace AutoForumSyncPro\Parser;

use Symfony\Component\DomCrawler\Crawler;

class PhpbbParser extends AbstractParser
{
    /**
     * PhpBB içeriği parse et
     */
    public function parse($html, $url)
    {
        $this->setCrawler($html);
        $this->setBaseUrl($url);
        
        $html = $this->cleanHtml($html);
        $this->setCrawler($html);
        
        $content = array(
            'title' => $this->extractTitle($html),
            'content' => $this->extractContent($html),
            'tags' => $this->extractTags($html),
            'unique_id' => $this->extractUniqueId($html, $url),
            'author' => $this->extractAuthor($html),
            'date' => $this->extractDate($html),
            'source_url' => $url
        );
        
        $this->validateContent($content);
        
        return $content;
    }

    /**
     * Başlık çıkar
     */
    protected function extractTitle($html)
    {
        // PhpBB topic title
        $title = $this->crawler->filter('h2.topic-title')->text();
        
        if (empty($title)) {
            $title = $this->crawler->filter('title')->text();
        }
        
        return trim($title);
    }

    /**
     * İçerik çıkar
     */
    protected function extractContent($html)
    {
        // PhpBB post content
        $content = $this->crawler->filter('.postbody .content')->html();
        
        if (empty($content)) {
            $content = $this->crawler->filter('.postbody')->html();
        }
        
        // URL'leri mutlak hale getir
        $content = $this->makeAbsoluteUrls($content, $this->baseUrl);
        
        // Resim URL'lerini optimize et
        $content = $this->optimizeImageUrls($content);
        
        return $content;
    }

    /**
     * Etiketleri çıkar
     */
    protected function extractTags($html)
    {
        $tags = array();
        
        // PhpBB tags (eğer varsa)
        $this->crawler->filter('.tag')->each(function($node) use (&$tags) {
            $tags[] = $node->text();
        });
        
        return $tags;
    }

    /**
     * Benzersiz ID çıkar
     */
    protected function extractUniqueId($html, $url)
    {
        // URL'den topic ID'yi çıkar
        if (preg_match('/t=(\d+)/', $url, $matches)) {
            return 'phpbb_topic_' . $matches[1];
        }
        
        // HTML'den topic ID'yi çıkar
        if (preg_match('/topic_id="(\d+)"/', $html, $matches)) {
            return 'phpbb_topic_' . $matches[1];
        }
        
        throw new \Exception('Benzersiz ID bulunamadı');
    }

    /**
     * Yazar bilgisi çıkar
     */
    protected function extractAuthor($html)
    {
        $author = $this->crawler->filter('.username')->text();
        
        if (empty($author)) {
            $author = $this->crawler->filter('.postauthor')->text();
        }
        
        return trim($author);
    }

    /**
     * Tarih bilgisi çıkar
     */
    protected function extractDate($html)
    {
        $dateText = $this->crawler->filter('.postdate')->text();
        
        if (empty($dateText)) {
            $dateText = $this->crawler->filter('.postdate time')->attr('datetime');
        }
        
        return $dateText;
    }
}
