<?php

namespace AutoForumSyncPro\Cron;

use XF\Cron\AbstractCron;
use AutoForumSyncPro\Service\ContentFetcher;
use AutoForumSyncPro\Service\ContentPublisher;
use AutoForumSyncPro\Service\Logger;
use AutoForumSyncPro\Entity\TargetSite;

class ContentSync extends AbstractCron
{
    protected $logger;
    protected $contentFetcher;
    protected $contentPublisher;

    public function __construct()
    {
        $this->logger = new Logger();
        $this->contentFetcher = new ContentFetcher();
        $this->contentPublisher = new ContentPublisher();
    }

    /**
     * Cron job çalıştır
     */
    public function run()
    {
        if (!\XF::options()->autoForumSyncEnabled) {
            return;
        }

        $this->logger->info("Otomatik içerik senkronizasyonu başlatıldı");

        try {
            // Aktif hedef siteleri al
            $targetSites = $this->getActiveTargetSites();
            
            foreach ($targetSites as $targetSite) {
                $this->processTargetSite($targetSite);
            }

            $this->logger->info("Otomatik içerik senkronizasyonu tamamlandı");

        } catch (\Exception $e) {
            $this->logger->error("Cron job hatası: " . $e->getMessage());
        }
    }

    /**
     * Hedef siteyi işle
     */
    protected function processTargetSite($targetSite)
    {
        try {
            $this->logger->info("Hedef site işleniyor: " . $targetSite['site_name']);

            // Bağlantı testi
            if (!$this->contentFetcher->testConnection($targetSite['base_url'])) {
                $this->logger->warning("Hedef site erişilemiyor: " . $targetSite['site_name']);
                return;
            }

            // Content fetcher'ı ayarla
            $targetSiteEntity = new TargetSite($targetSite);
            $this->contentFetcher->setTargetSite($targetSiteEntity);

            // URL'leri al
            $urls = $this->getUrlsToFetch($targetSite);
            
            if (empty($urls)) {
                $this->logger->info("İşlenecek URL bulunamadı: " . $targetSite['site_name']);
                return;
            }

            // İçerikleri çek
            $contents = $this->contentFetcher->fetchMultiple($urls);
            
            if (empty($contents)) {
                $this->logger->info("Çekilen içerik bulunamadı: " . $targetSite['site_name']);
                return;
            }

            // İçerikleri yayınla
            $options = array(
                'user_id' => $targetSite['default_user_id'],
                'forum_id' => $targetSite['default_forum_id']
            );

            $results = $this->contentPublisher->publishMultiple($contents, $options);
            
            $this->logger->info("Başarıyla yayınlanan içerik sayısı: " . count($results));

            // Son çalışma zamanını güncelle
            $this->updateLastRunTime($targetSite['target_id']);

        } catch (\Exception $e) {
            $this->logger->error("Hedef site işleme hatası: " . $e->getMessage());
        }
    }

    /**
     * Aktif hedef siteleri al
     */
    protected function getActiveTargetSites()
    {
        $sites = \XF::db()->fetchAll(
            "SELECT * FROM xf_auto_forum_sync_targets WHERE enabled = 1"
        );

        return $sites;
    }

    /**
     * Çekilecek URL'leri al
     */
    protected function getUrlsToFetch($targetSite)
    {
        $urls = array();

        // Hedef site tipine göre URL'leri al
        switch ($targetSite['site_type']) {
            case 'xenforo':
                $urls = $this->getXenForoUrls($targetSite);
                break;
            case 'vbulletin':
                $urls = $this->getVBulletinUrls($targetSite);
                break;
            case 'phpbb':
                $urls = $this->getPhpBBUrls($targetSite);
                break;
            case 'custom':
                $urls = $this->getCustomUrls($targetSite);
                break;
        }

        return $urls;
    }

    /**
     * XenForo URL'lerini al
     */
    protected function getXenForoUrls($targetSite)
    {
        // XenForo API veya scraping ile URL'leri al
        $baseUrl = rtrim($targetSite['base_url'], '/');
        $forumUrl = $baseUrl . '/forums/';
        
        // Son çalışma zamanından sonraki konuları al
        $lastRun = $targetSite['last_run_time'];
        
        // Bu örnekte basit bir URL listesi döndürüyoruz
        // Gerçek implementasyonda XenForo API'sini kullanabilirsiniz
        return array(
            $baseUrl . '/threads/sample-thread-1.123/',
            $baseUrl . '/threads/sample-thread-2.124/',
        );
    }

    /**
     * VBulletin URL'lerini al
     */
    protected function getVBulletinUrls($targetSite)
    {
        $baseUrl = rtrim($targetSite['base_url'], '/');
        
        return array(
            $baseUrl . '/showthread.php?t=123',
            $baseUrl . '/showthread.php?t=124',
        );
    }

    /**
     * PhpBB URL'lerini al
     */
    protected function getPhpBBUrls($targetSite)
    {
        $baseUrl = rtrim($targetSite['base_url'], '/');
        
        return array(
            $baseUrl . '/viewtopic.php?t=123',
            $baseUrl . '/viewtopic.php?t=124',
        );
    }

    /**
     * Özel URL'leri al
     */
    protected function getCustomUrls($targetSite)
    {
        // Özel URL listesi veya RSS feed'den URL'leri al
        $urls = $targetSite['custom_urls'];
        
        if (empty($urls)) {
            return array();
        }

        return is_array($urls) ? $urls : explode("\n", $urls);
    }

    /**
     * Son çalışma zamanını güncelle
     */
    protected function updateLastRunTime($targetId)
    {
        \XF::db()->update(
            'xf_auto_forum_sync_targets',
            array('last_run_time' => \XF::$time),
            'target_id = ?',
            $targetId
        );
    }

    /**
     * Manuel çalıştırma
     */
    public function runManual($targetSiteId = null)
    {
        if ($targetSiteId) {
            $site = \XF::db()->fetchRow(
                "SELECT * FROM xf_auto_forum_sync_targets WHERE target_id = ?",
                array($targetSiteId)
            );
            
            if ($site) {
                $this->processTargetSite($site);
            }
        } else {
            $this->run();
        }
    }

    /**
     * Zamanlanmış çalıştırma
     */
    public function runScheduled()
    {
        $scheduledRuns = \XF::db()->fetchAll(
            "SELECT * FROM xf_auto_forum_sync_schedules WHERE enabled = 1 AND next_run <= ?",
            array(\XF::$time)
        );

        foreach ($scheduledRuns as $schedule) {
            $this->runManual($schedule['target_id']);
            
            // Sonraki çalışma zamanını hesapla
            $nextRun = $this->calculateNextRun($schedule);
            
            \XF::db()->update(
                'xf_auto_forum_sync_schedules',
                array('next_run' => $nextRun),
                'schedule_id = ?',
                $schedule['schedule_id']
            );
        }
    }

    /**
     * Sonraki çalışma zamanını hesapla
     */
    protected function calculateNextRun($schedule)
    {
        $interval = $schedule['interval_minutes'];
        return \XF::$time + ($interval * 60);
    }
}
