<?php
namespace XenSoluce\SEOOptimization\SEO;

use XF\Mvc\Entity\Entity;
use XF\Template\Templater;

abstract class AbstractHandler
{
    protected string $contentType = '';
    protected Entity $entity;
    protected string $nameItemTitle = '';
    protected string $nameItemDescription = '';
    protected Templater $templater;
    protected \ArrayObject $optionSEO;

    abstract protected function changeOpenGraph($default = '');
    abstract protected function option(): array;
    abstract protected function description(): string;

    public function __construct($contentType)
    {
        $this->contentType = $contentType;

        $this->constructOption();
    }

    /**
     *
     */
    protected function constructOption(): void
    {
        $defaultOption = [
            'separator' => '-',
            'metaTitle' => '',
            'metaDesc' => ''
        ];

        $this->optionSEO = new \ArrayObject($this->option() + $defaultOption, \ArrayObject::ARRAY_AS_PROPS);
    }

    /**
     * @param Entity $entity
     */
    public function setEntity(Entity $entity): void
    {
        $this->entity = $entity;
    }

    /**
     * @param Templater $templater
     */
    public function setTemplater(Templater $templater): void
    {
        $this->templater = $templater;
    }

    protected function getSegmentArray(): array
    {
        return [
            'item_title' => $this->entity->title,
            'category_title' => !empty($this->entity->Category->title) ? $this->entity->Category->title : '',
            'board_title' => \XF::options()->boardTitle,
        ];
    }

    public function changeTitle($escape, array $entityOption = []): ?string
    {
        $escape = true;
        if(!empty($this->entity->xs_seo['meta_title']))
        {
            return $this->templater->escape($this->entity->xs_seo['meta_title'], $escape);
        }

        if (isset($this->entity->title) && strlen($this->entity->title))
        {
            $segmentsArray = [];
            if($this->nameItemTitle != "")
            {
                $prefixTitle = $this->templater->fnPrefix(
                    $this->templater,
                    $escape,
                    $this->nameItemTitle,
                    $this->entity,
                    'plain',
                    ''
                );
                $segmentsArray['prefix_title'] = $prefixTitle;
            }

            $newEntityTitle = $this->entity->title . ' | ' . \XF::options()->boardTitle;
            $newEntityTitleFormat = $this->optionSEO->metaTitle;

            if(!empty($newEntityTitleFormat))
            {
                $separator = $this->optionSEO->separator;
                $newEntityTitleFormat = preg_replace('/(%)\s(%)/', '$1$2', $newEntityTitleFormat);
                $titleSegments = @explode('%', $newEntityTitleFormat);

                if($titleSegments)
                {
                    $segmentsArray += $this->getSegmentArray();

                    $titleSegments = str_replace(array_keys($segmentsArray), array_values($segmentsArray), $titleSegments);
                    $titleSegments = array_filter($titleSegments);
                    $newEntityTitle = $separator == 'none'
                        ? implode(" ", $titleSegments)
                        : implode(" $separator ", $titleSegments);
                }
            }

            return $this->templater->escape($newEntityTitle, $escape);
        }

        return sprintf('%s | %s',
            $this->templater->escape($this->entity->title, $escape),
            $this->templater->escape(\XF::options()->boardTitle, $escape)
        );
    }

    public function getOpenGraph($default = '')
    {
        if(!empty($this->entity->xs_seo['open_graph']))
        {
            return $this->entity->xs_seo['open_graph'];
        }

        return $this->changeOpenGraph($default);
    }

    public function changeDescription(&$escape)
    {
        $escape = true;
        $options = \XF::options();
        if($this->optionSEO->metaDesc)
        {
            $categoryTitle = '';
            if(!empty($this->entity->Category->title))
            {
                $categoryTitle = $this->entity->Category->title;
            }

            $segmentsArray = [];
            if($this->nameItemDescription != "")
            {
                $prefixTitle = $this->templater->fnPrefix($this->templater, $escape, $this->nameItemDescription, $this->entity, 'plain', '');
                $segmentsArray['prefix_title'] = $prefixTitle;
            }

            $entityDesc = preg_replace('/(%)\s(%)/', '$1$2', $this->optionSEO->metaDesc);
            $descSegments = @explode('%', $entityDesc);

            if($descSegments)
            {
                $segmentsArray += [
                    'item_title' => $this->templater->escape($this->entity->title, $escape),
                    'category_title' => $categoryTitle,
                    'board_title' => $options->boardTitle,
                ];

                $descSegments = str_replace(array_keys($segmentsArray), array_values($segmentsArray), $descSegments);
                $descSegments = array_filter($descSegments);
                $entityDesc = implode("", $descSegments);
            }

            return $entityDesc;
        }
        else
        {
            return $this->templater->fnSnippet($this->templater, $escape, $this->description(), 0, ['stripBbCode' => true]);
        }
    }

    public function getAttachment($attachments, $default, $canViewAttachment)
    {
        if(!empty($attachments))
        {
            foreach($attachments as $attachment)
            {
                if($attachment->has_thumbnail)
                {
                    $imageUrl = rtrim(\XF::app()->request()->getHostUrl(), '/') . '/' . ltrim($attachment->thumbnail_url, '/');

                    if($canViewAttachment)
                    {
                        $imageUrl = \XF::app()->router()->buildLink('full:attachments', $attachment);
                    }

                    return $imageUrl;
                }
            }
        }

        return $default;
    }

    protected function buildLink($link)
    {
        return rtrim(\XF::app()->request()->getHostUrl(), '/') . '/' . ltrim($link, '/');
    }

    protected function getAvatarUrl($default = '')
    {
        if(!empty($this->entity->User) && $this->entity->User instanceof \XF\Entity\User)
        {
            $avatarUrl = $this->entity->User->getAvatarUrl('h', '', true);
            if($avatarUrl)
            {
                return $avatarUrl;
            }
        }

        return $default;
    }

    protected function boardTitleLength($title)
    {
        $options = \XF::options();
        $boardTitle = $options->boardTitle;
        if($options->xs_seo_medium_title_length > 0 AND !empty($options->xs_seo_medium_title))
        {
            if(strlen($title) > $options->xs_seo_medium_title_length)
            {
                $boardTitle = $options->xs_seo_medium_title;
            }
        }
        if($options->xs_seo_small_title_length > 0)
        {
            if(strlen($title) > $options->xs_seo_small_title_length)
            {
                $boardTitle = '';
            }
        }

        return $boardTitle;
    }
}