<?php

namespace AutoForumSyncPro\Service;

class SpamFilter
{
    protected $blacklistWords = [];
    protected $whitelistWords = [];
    protected $regexPatterns = [];
    protected $htmlElementsToRemove = [];

    public function __construct()
    {
        $this->loadFilters();
    }

    /**
     * Spam filtrelerini yükle
     */
    protected function loadFilters()
    {
        // Veritabanından filtreleri yükle
        $this->blacklistWords = $this->getBlacklistWords();
        $this->whitelistWords = $this->getWhitelistWords();
        $this->regexPatterns = $this->getRegexPatterns();
        $this->htmlElementsToRemove = $this->getHtmlElementsToRemove();
    }

    /**
     * İçeriğin spam olup olmadığını kontrol et
     */
    public function isSpam($content)
    {
        $text = $this->extractText($content);
        
        // Kara liste kontrolü
        if ($this->checkBlacklist($text)) {
            return true;
        }

        // Beyaz liste kontrolü (eğer beyaz liste varsa)
        if (!empty($this->whitelistWords) && !$this->checkWhitelist($text)) {
            return true;
        }

        // Regex kontrolü
        if ($this->checkRegexPatterns($text)) {
            return true;
        }

        // HTML element kontrolü
        if ($this->checkHtmlElements($content)) {
            return true;
        }

        return false;
    }

    /**
     * Metin çıkar
     */
    protected function extractText($content)
    {
        $text = '';
        
        if (isset($content['title'])) {
            $text .= ' ' . $content['title'];
        }
        
        if (isset($content['content'])) {
            $text .= ' ' . strip_tags($content['content']);
        }
        
        if (isset($content['tags']) && is_array($content['tags'])) {
            $text .= ' ' . implode(' ', $content['tags']);
        }

        return strtolower(trim($text));
    }

    /**
     * Kara liste kontrolü
     */
    protected function checkBlacklist($text)
    {
        foreach ($this->blacklistWords as $word) {
            if (strpos($text, strtolower($word)) !== false) {
                return true;
            }
        }
        return false;
    }

    /**
     * Beyaz liste kontrolü
     */
    protected function checkWhitelist($text)
    {
        foreach ($this->whitelistWords as $word) {
            if (strpos($text, strtolower($word)) !== false) {
                return true;
            }
        }
        return false;
    }

    /**
     * Regex pattern kontrolü
     */
    protected function checkRegexPatterns($text)
    {
        foreach ($this->regexPatterns as $pattern) {
            if (preg_match($pattern, $text)) {
                return true;
            }
        }
        return false;
    }

    /**
     * HTML element kontrolü
     */
    protected function checkHtmlElements($content)
    {
        if (!isset($content['content'])) {
            return false;
        }

        $html = $content['content'];
        
        foreach ($this->htmlElementsToRemove as $element) {
            if (strpos($html, $element) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * İçeriği temizle
     */
    public function clean($content)
    {
        if (!isset($content['content'])) {
            return $content;
        }

        $html = $content['content'];

        // HTML elementlerini kaldır
        foreach ($this->htmlElementsToRemove as $element) {
            $html = preg_replace('/<' . preg_quote($element) . '[^>]*>.*?<\/' . preg_quote($element) . '>/is', '', $html);
        }

        // Fazla boşlukları temizle
        $html = preg_replace('/\s+/', ' ', $html);
        $html = trim($html);

        $content['content'] = $html;
        return $content;
    }

    /**
     * Kara liste kelimelerini al
     */
    protected function getBlacklistWords()
    {
        $words = \XF::db()->fetchAll(
            "SELECT word FROM xf_auto_forum_sync_blacklist WHERE type = 'word' AND enabled = 1"
        );
        
        $result = [];
        foreach ($words as $word) {
            $result[] = $word['word'];
        }
        
        return $result;
    }

    /**
     * Beyaz liste kelimelerini al
     */
    protected function getWhitelistWords()
    {
        $words = \XF::db()->fetchAll(
            "SELECT word FROM xf_auto_forum_sync_whitelist WHERE enabled = 1"
        );
        
        $result = [];
        foreach ($words as $word) {
            $result[] = $word['word'];
        }
        
        return $result;
    }

    /**
     * Regex pattern'lerini al
     */
    protected function getRegexPatterns()
    {
        $patterns = \XF::db()->fetchAll(
            "SELECT pattern FROM xf_auto_forum_sync_blacklist WHERE type = 'regex' AND enabled = 1"
        );
        
        $result = [];
        foreach ($patterns as $pattern) {
            $result[] = $pattern['pattern'];
        }
        
        return $result;
    }

    /**
     * Kaldırılacak HTML elementlerini al
     */
    protected function getHtmlElementsToRemove()
    {
        $elements = \XF::db()->fetchAll(
            "SELECT element FROM xf_auto_forum_sync_html_filters WHERE enabled = 1"
        );
        
        $result = [];
        foreach ($elements as $element) {
            $result[] = $element['element'];
        }
        
        return $result;
    }

    /**
     * Kara liste kelimesi ekle
     */
    public function addBlacklistWord($word, $type = 'word')
    {
        \XF::db()->insert('xf_auto_forum_sync_blacklist', [
            'word' => $word,
            'type' => $type,
            'enabled' => 1,
            'created_date' => \XF::$time
        ]);
    }

    /**
     * Beyaz liste kelimesi ekle
     */
    public function addWhitelistWord($word)
    {
        \XF::db()->insert('xf_auto_forum_sync_whitelist', [
            'word' => $word,
            'enabled' => 1,
            'created_date' => \XF::$time
        ]);
    }

    /**
     * HTML element filtresi ekle
     */
    public function addHtmlElementFilter($element)
    {
        \XF::db()->insert('xf_auto_forum_sync_html_filters', [
            'element' => $element,
            'enabled' => 1,
            'created_date' => \XF::$time
        ]);
    }
}
