<?php

namespace AutoForumSyncPro\Service;

class KeywordReplacer
{
    protected $replacements = [];
    protected $regexReplacements = [];

    public function __construct()
    {
        $this->loadReplacements();
    }

    /**
     * Anahtar kelime değiştirmelerini yükle
     */
    protected function loadReplacements()
    {
        $this->replacements = $this->getSimpleReplacements();
        $this->regexReplacements = $this->getRegexReplacements();
    }

    /**
     * İçerikte anahtar kelime değiştirmelerini uygula
     */
    public function replace($content)
    {
        // Başlık değiştirmeleri
        if (isset($content['title'])) {
            $content['title'] = $this->replaceInText($content['title']);
        }

        // İçerik değiştirmeleri
        if (isset($content['content'])) {
            $content['content'] = $this->replaceInText($content['content']);
        }

        // Etiket değiştirmeleri
        if (isset($content['tags']) && is_array($content['tags'])) {
            $content['tags'] = array_map([$this, 'replaceInText'], $content['tags']);
        }

        return $content;
    }

    /**
     * Metinde değiştirmeleri uygula
     */
    protected function replaceInText($text)
    {
        // Basit değiştirmeler
        foreach ($this->replacements as $search => $replace) {
            $text = str_ireplace($search, $replace, $text);
        }

        // Regex değiştirmeleri
        foreach ($this->regexReplacements as $pattern => $replacement) {
            $text = preg_replace($pattern, $replacement, $text);
        }

        return $text;
    }

    /**
     * Basit değiştirmeleri al
     */
    protected function getSimpleReplacements()
    {
        $replacements = \XF::db()->fetchAll(
            "SELECT search_text, replace_text FROM xf_auto_forum_sync_replacements 
             WHERE type = 'simple' AND enabled = 1"
        );
        
        $result = [];
        foreach ($replacements as $replacement) {
            $result[$replacement['search_text']] = $replacement['replace_text'];
        }
        
        return $result;
    }

    /**
     * Regex değiştirmelerini al
     */
    protected function getRegexReplacements()
    {
        $replacements = \XF::db()->fetchAll(
            "SELECT search_text, replace_text FROM xf_auto_forum_sync_replacements 
             WHERE type = 'regex' AND enabled = 1"
        );
        
        $result = [];
        foreach ($replacements as $replacement) {
            $result[$replacement['search_text']] = $replacement['replace_text'];
        }
        
        return $result;
    }

    /**
     * Yeni değiştirme kuralı ekle
     */
    public function addReplacement($searchText, $replaceText, $type = 'simple')
    {
        \XF::db()->insert('xf_auto_forum_sync_replacements', [
            'search_text' => $searchText,
            'replace_text' => $replaceText,
            'type' => $type,
            'enabled' => 1,
            'created_date' => \XF::$time
        ]);
    }

    /**
     * Değiştirme kuralını güncelle
     */
    public function updateReplacement($id, $searchText, $replaceText, $enabled = 1)
    {
        \XF::db()->update('xf_auto_forum_sync_replacements', [
            'search_text' => $searchText,
            'replace_text' => $replaceText,
            'enabled' => $enabled
        ], 'replacement_id = ?', $id);
    }

    /**
     * Değiştirme kuralını sil
     */
    public function deleteReplacement($id)
    {
        \XF::db()->delete('xf_auto_forum_sync_replacements', 'replacement_id = ?', $id);
    }

    /**
     * Tüm değiştirmeleri yeniden yükle
     */
    public function reloadReplacements()
    {
        $this->loadReplacements();
    }

    /**
     * Test değiştirmesi
     */
    public function testReplacement($text, $searchText, $replaceText, $type = 'simple')
    {
        if ($type === 'regex') {
            return preg_replace($searchText, $replaceText, $text);
        } else {
            return str_ireplace($searchText, $replaceText, $text);
        }
    }

    /**
     * URL değiştirmeleri
     */
    public function replaceUrls($content)
    {
        if (!isset($content['content'])) {
            return $content;
        }

        $text = $content['content'];
        
        // URL'leri bul ve değiştir
        $urlReplacements = $this->getUrlReplacements();
        
        foreach ($urlReplacements as $oldUrl => $newUrl) {
            $text = str_replace($oldUrl, $newUrl, $text);
        }

        $content['content'] = $text;
        return $content;
    }

    /**
     * URL değiştirmelerini al
     */
    protected function getUrlReplacements()
    {
        $replacements = \XF::db()->fetchAll(
            "SELECT search_text, replace_text FROM xf_auto_forum_sync_replacements 
             WHERE type = 'url' AND enabled = 1"
        );
        
        $result = [];
        foreach ($replacements as $replacement) {
            $result[$replacement['search_text']] = $replacement['replace_text'];
        }
        
        return $result;
    }

    /**
     * Markdown/BBCode dönüştürme
     */
    public function convertFormatting($content)
    {
        if (!isset($content['content'])) {
            return $content;
        }

        $text = $content['content'];
        
        // BBCode'u HTML'e çevir
        $text = $this->bbcodeToHtml($text);
        
        // HTML'i BBCode'a çevir (XenForo için)
        $text = $this->htmlToBbcode($text);

        $content['content'] = $text;
        return $content;
    }

    /**
     * BBCode'u HTML'e çevir
     */
    protected function bbcodeToHtml($text)
    {
        $patterns = [
            '/\[b\](.*?)\[\/b\]/is' => '<strong>$1</strong>',
            '/\[i\](.*?)\[\/i\]/is' => '<em>$1</em>',
            '/\[u\](.*?)\[\/u\]/is' => '<u>$1</u>',
            '/\[url\](.*?)\[\/url\]/is' => '<a href="$1">$1</a>',
            '/\[url=(.*?)\](.*?)\[\/url\]/is' => '<a href="$1">$2</a>',
            '/\[img\](.*?)\[\/img\]/is' => '<img src="$1" alt="">',
        ];

        return preg_replace(array_keys($patterns), array_values($patterns), $text);
    }

    /**
     * HTML'i BBCode'a çevir
     */
    protected function htmlToBbcode($text)
    {
        $patterns = [
            '/<strong>(.*?)<\/strong>/is' => '[b]$1[/b]',
            '/<em>(.*?)<\/em>/is' => '[i]$1[/i]',
            '/<u>(.*?)<\/u>/is' => '[u]$1[/u]',
            '/<a href="(.*?)">(.*?)<\/a>/is' => '[url=$1]$2[/url]',
            '/<img src="(.*?)" alt="(.*?)">/is' => '[img]$1[/img]',
        ];

        return preg_replace(array_keys($patterns), array_values($patterns), $text);
    }
}
