<?php

namespace AutoForumSyncPro\Service;

use XF\Entity\User;
use XF\Entity\Forum;
use XF\Entity\Thread;
use XF\Entity\Post;
use AutoForumSyncPro\Service\Logger;

class ContentPublisher
{
    protected $logger;
    protected $defaultUserId;
    protected $defaultForumId;

    public function __construct()
    {
        $this->logger = new Logger();
        $this->defaultUserId = \XF::options()->autoForumSyncDefaultUser;
        $this->defaultForumId = \XF::options()->autoForumSyncDefaultForum;
    }

    /**
     * İçeriği forumda yayınla
     */
    public function publish($content, $options = [])
    {
        try {
            // Kullanıcı ve forum ID'lerini belirle
            $userId = isset($options['user_id']) ? $options['user_id'] : $this->defaultUserId;
            $forumId = isset($options['forum_id']) ? $options['forum_id'] : $this->defaultForumId;
            
            // Kullanıcı kontrolü
            $user = $this->getUser($userId);
            if (!$user) {
                throw new \Exception("Geçersiz kullanıcı ID: {$userId}");
            }

            // Forum kontrolü
            $forum = $this->getForum($forumId);
            if (!$forum) {
                throw new \Exception("Geçersiz forum ID: {$forumId}");
            }

            // Thread oluştur
            $thread = $this->createThread($content, $user, $forum, $options);
            
            // İlk post'u oluştur
            $post = $this->createPost($content, $thread, $user);
            
            // Log kaydı
            $this->logContent($content, $thread, $post);
            
            $this->logger->info("İçerik başarıyla yayınlandı: Thread ID {$thread->thread_id}");
            
            return [
                'thread' => $thread,
                'post' => $post
            ];

        } catch (\Exception $e) {
            $this->logger->error("Yayınlama hatası: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Thread oluştur
     */
    protected function createThread($content, User $user, Forum $forum, $options)
    {
        $threadData = array(
            'title' => $content['title'],
            'user_id' => $user->user_id,
            'username' => $user->username,
            'node_id' => $forum->node_id,
            'discussion_state' => 'visible',
            'discussion_open' => 1,
            'sticky' => 0,
            'tags' => isset($content['tags']) ? $content['tags'] : array()
        );

        $thread = $forum->getNewThread();
        $thread->bulkSet($threadData);
        $thread->save();

        return $thread;
    }

    /**
     * Post oluştur
     */
    protected function createPost($content, Thread $thread, User $user)
    {
        $postData = array(
            'thread_id' => $thread->thread_id,
            'user_id' => $user->user_id,
            'username' => $user->username,
            'post_date' => \XF::$time,
            'message' => $content['content'],
            'message_state' => 'visible',
            'position' => 0
        );

        $post = $thread->getNewPost();
        $post->bulkSet($postData);
        $post->save();

        return $post;
    }

    /**
     * Kullanıcı al
     */
    protected function getUser($userId)
    {
        if ($userId > 0) {
            return \XF::em()->find('XF:User', $userId);
        }

        // Varsayılan kullanıcı oluştur
        return $this->createDefaultUser();
    }

    /**
     * Varsayılan kullanıcı oluştur
     */
    protected function createDefaultUser()
    {
        $username = 'AutoSync_' . uniqid();
        
        $user = \XF::em()->create('XF:User');
        $user->bulkSet(array(
            'username' => $username,
            'email' => $username . '@autosync.local',
            'user_state' => 'valid',
            'is_staff' => 0,
            'user_group_id' => 2 // Registered user group
        ));
        
        $user->save();
        
        return $user;
    }

    /**
     * Forum al
     */
    protected function getForum($forumId)
    {
        return \XF::em()->find('XF:Forum', $forumId);
    }

    /**
     * İçerik logla
     */
    protected function logContent($content, Thread $thread, Post $post)
    {
        $logData = array(
            'unique_id' => isset($content['unique_id']) ? $content['unique_id'] : null,
            'source_url' => isset($content['source_url']) ? $content['source_url'] : null,
            'thread_id' => $thread->thread_id,
            'post_id' => $post->post_id,
            'user_id' => $thread->user_id,
            'created_date' => \XF::$time,
            'status' => 'published'
        );

        \XF::db()->insert('xf_auto_forum_sync_content', $logData);
    }

    /**
     * Toplu yayınlama
     */
    public function publishMultiple($contents, $options = [])
    {
        $results = [];
        $maxPosts = \XF::options()->autoForumSyncMaxPostsPerRun;
        $delay = \XF::options()->autoForumSyncDelayBetweenPosts;

        $count = 0;
        foreach ($contents as $content) {
            if ($count >= $maxPosts) {
                break;
            }

            $result = $this->publish($content, $options);
            if ($result) {
                $results[] = $result;
                $count++;
            }

            // Rate limiting
            if ($delay > 0) {
                sleep($delay);
            }
        }

        return $results;
    }

    /**
     * İçerik onay sistemi
     */
    public function publishWithApproval($content, $options = [])
    {
        // Onay gerektiren durumlarda içeriği bekleme durumuna al
        $content['status'] = 'pending_approval';
        
        // Veritabanına kaydet
        $this->savePendingContent($content, $options);
        
        $this->logger->info("İçerik onay için beklemede: " . $content['title']);
        
        return true;
    }

    /**
     * Bekleyen içeriği kaydet
     */
    protected function savePendingContent($content, $options)
    {
        $data = array(
            'unique_id' => isset($content['unique_id']) ? $content['unique_id'] : null,
            'source_url' => isset($content['source_url']) ? $content['source_url'] : null,
            'title' => $content['title'],
            'content' => $content['content'],
            'tags' => json_encode(isset($content['tags']) ? $content['tags'] : array()),
            'user_id' => isset($options['user_id']) ? $options['user_id'] : $this->defaultUserId,
            'forum_id' => isset($options['forum_id']) ? $options['forum_id'] : $this->defaultForumId,
            'status' => 'pending_approval',
            'created_date' => \XF::$time
        );

        \XF::db()->insert('xf_auto_forum_sync_pending', $data);
    }
}
