<?php

namespace AutoForumSyncPro\Service;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\DomCrawler\Crawler;
use AutoForumSyncPro\Entity\TargetSite;
use AutoForumSyncPro\Service\SpamFilter;
use AutoForumSyncPro\Service\KeywordReplacer;
use AutoForumSyncPro\Service\Logger;

class ContentFetcher
{
    protected $httpClient;
    protected $spamFilter;
    protected $keywordReplacer;
    protected $logger;
    protected $targetSite;
    protected $parser;

    public function __construct()
    {
        $this->httpClient = new Client(array(
            'timeout' => 30,
            'connect_timeout' => 10,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
            )
        ));
        
        $this->spamFilter = new SpamFilter();
        $this->keywordReplacer = new KeywordReplacer();
        $this->logger = new Logger();
    }

    /**
     * Hedef siteyi ayarla ve parser'ı başlat
     */
    public function setTargetSite(TargetSite $targetSite)
    {
        $this->targetSite = $targetSite;
        $this->parser = $this->createParser($targetSite->getSiteType());
        return $this;
    }

    /**
     * URL'den içerik çek
     */
    public function fetchFromUrl($url)
    {
        try {
            $this->logger->info("İçerik çekiliyor: {$url}");
            
            // URL geçerliliğini kontrol et
            if (!$this->isValidUrl($url)) {
                throw new \Exception("Geçersiz URL: {$url}");
            }

            // HTTP isteği gönder
            $response = $this->httpClient->get($url);
            $html = $response->getBody()->getContents();
            
            // Parser ile içeriği ayrıştır
            $content = $this->parser->parse($html, $url);
            
            // Spam filtresini uygula
            if ($this->spamFilter->isSpam($content)) {
                $this->logger->warning("Spam içerik tespit edildi: {$url}");
                return null;
            }

            // Anahtar kelime değiştirmelerini uygula
            $content = $this->keywordReplacer->replace($content);
            
            // Duplicate kontrolü
            if ($this->isDuplicate($content)) {
                $this->logger->info("Duplicate içerik tespit edildi: {$url}");
                return null;
            }

            $this->logger->info("İçerik başarıyla çekildi: {$url}");
            return $content;

        } catch (RequestException $e) {
            $this->logger->error("HTTP hatası: " . $e->getMessage());
            return null;
        } catch (\Exception $e) {
            $this->logger->error("Genel hata: " . $e->getMessage());
            return null;
        }
    }

    /**
     * URL geçerliliğini kontrol et
     */
    protected function isValidUrl($url)
    {
        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }

    /**
     * Duplicate kontrolü
     */
    protected function isDuplicate($content)
    {
        // Benzersiz ID veya hash kontrolü
        $uniqueId = isset($content['unique_id']) ? $content['unique_id'] : null;
        if (!$uniqueId) {
            return false;
        }

        // Veritabanında kontrol et
        $existing = \XF::db()->fetchOne(
            "SELECT content_id FROM xf_auto_forum_sync_content WHERE unique_id = ?",
            [$uniqueId]
        );

        return $existing !== false;
    }

    /**
     * Site tipine göre parser oluştur
     */
    protected function createParser($siteType)
    {
        $parserClass = "AutoForumSyncPro\\Parser\\" . ucfirst($siteType) . "Parser";
        
        if (!class_exists($parserClass)) {
            throw new \Exception("Parser sınıfı bulunamadı: {$parserClass}");
        }

        return new $parserClass();
    }

    /**
     * Hedef site erişilebilirliğini test et
     */
    public function testConnection($url)
    {
        try {
            $response = $this->httpClient->get($url, array('timeout' => 10));
            return $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Toplu içerik çekme
     */
    public function fetchMultiple($urls)
    {
        $results = [];
        
        foreach ($urls as $url) {
            $content = $this->fetchFromUrl($url);
            if ($content) {
                $results[] = $content;
            }
            
            // Rate limiting
            sleep($this->targetSite->getDelayBetweenRequests());
        }
        
        return $results;
    }
}
