<?php

namespace AutoForumSyncPro\Parser;

use Symfony\Component\DomCrawler\Crawler;

abstract class AbstractParser
{
    protected $crawler;
    protected $baseUrl;

    /**
     * HTML içeriği parse et
     */
    abstract public function parse($html, $url);

    /**
     * Başlık çıkar
     */
    abstract protected function extractTitle($html);

    /**
     * İçerik çıkar
     */
    abstract protected function extractContent($html);

    /**
     * Etiketleri çıkar
     */
    abstract protected function extractTags($html);

    /**
     * Benzersiz ID çıkar
     */
    abstract protected function extractUniqueId($html, $url);

    /**
     * Yazar bilgisi çıkar
     */
    abstract protected function extractAuthor($html);

    /**
     * Tarih bilgisi çıkar
     */
    abstract protected function extractDate($html);

    /**
     * Crawler'ı ayarla
     */
    protected function setCrawler($html)
    {
        $this->crawler = new Crawler($html);
    }

    /**
     * Base URL'i ayarla
     */
    protected function setBaseUrl($url)
    {
        $this->baseUrl = $url;
    }

    /**
     * HTML'i temizle
     */
    protected function cleanHtml($html)
    {
        // Gereksiz script ve style taglarını kaldır
        $html = preg_replace('/<script[^>]*>.*?<\/script>/is', '', $html);
        $html = preg_replace('/<style[^>]*>.*?<\/style>/is', '', $html);
        
        // Reklam div'lerini kaldır
        $html = preg_replace('/<div[^>]*class="[^"]*ad[^"]*"[^>]*>.*?<\/div>/is', '', $html);
        
        return $html;
    }

    /**
     * Metin çıkar
     */
    protected function extractText($html)
    {
        return strip_tags($html);
    }

    /**
     * URL'leri mutlak hale getir
     */
    protected function makeAbsoluteUrls($html, $baseUrl)
    {
        $baseUrl = rtrim($baseUrl, '/');
        
        // Relative URL'leri mutlak hale getir
        $html = preg_replace('/src="\/([^"]*)"/', 'src="' . $baseUrl . '/$1"', $html);
        $html = preg_replace('/href="\/([^"]*)"/', 'href="' . $baseUrl . '/$1"', $html);
        
        return $html;
    }

    /**
     * Resim URL'lerini optimize et
     */
    protected function optimizeImageUrls($html)
    {
        // Resim URL'lerini bul ve optimize et
        preg_match_all('/<img[^>]+src="([^"]+)"[^>]*>/i', $html, $matches);
        
        foreach ($matches[1] as $imageUrl) {
            // Resim boyutlarını kontrol et ve gerekirse optimize et
            $optimizedUrl = $this->optimizeImageUrl($imageUrl);
            $html = str_replace($imageUrl, $optimizedUrl, $html);
        }
        
        return $html;
    }

    /**
     * Resim URL'ini optimize et
     */
    protected function optimizeImageUrl($url)
    {
        // Burada resim optimizasyonu yapılabilir
        // Örneğin: boyut küçültme, format dönüştürme vb.
        return $url;
    }

    /**
     * İçeriği doğrula
     */
    protected function validateContent($content)
    {
        if (empty($content['title'])) {
            throw new \Exception('Başlık bulunamadı');
        }
        
        if (empty($content['content'])) {
            throw new \Exception('İçerik bulunamadı');
        }
        
        if (empty($content['unique_id'])) {
            throw new \Exception('Benzersiz ID bulunamadı');
        }
        
        return true;
    }
}
