# AutoForumSync Pro - Kullanım Kılavuzu

## 🚀 Hızlı Başlangıç

### 1. İlk Hedef Site Ekleme

1. **Admin Panel** → **Eklentiler** → **AutoForumSync Pro** → **Hedef Siteler**
2. **Yeni Hedef Site** butonuna tıklayın
3. Aşağıdaki bilgileri doldurun:

```
Site Adı: Örnek Forum
Base URL: https://example-forum.com
Site Tipi: XenForo
Varsayılan Kullanıcı: [Bir kullanıcı seçin]
Varsayılan Forum: [Bir forum seçin]
İstekler Arası Gecikme: 2 saniye
```

4. **Kaydet** butonuna tıklayın

### 2. Test Bağlantısı

1. Hedef site listesinde **Test** butonuna tıklayın
2. Bağlantı başarılı ise yeşil onay işareti görünür
3. Başarısız ise hata mesajını kontrol edin

### 3. İlk İçerik Çekme

1. **Manuel Senkronizasyon** butonuna tıklayın
2. Log kayıtlarını kontrol edin
3. Yayınlanan içerikleri forumda görün

## ⚙️ Detaylı Yapılandırma

### Hedef Site Tipleri

#### XenForo Forumları
```
Base URL: https://xenforo-site.com
Site Tipi: XenForo
```

**Desteklenen URL Formatları:**
- `https://site.com/threads/thread-title.123/`
- `https://site.com/forums/forum-name.456/`

#### VBulletin Forumları
```
Base URL: https://vbulletin-site.com
Site Tipi: VBulletin
```

**Desteklenen URL Formatları:**
- `https://site.com/showthread.php?t=123`
- `https://site.com/forumdisplay.php?f=456`

#### PhpBB Forumları
```
Base URL: https://phpbb-site.com
Site Tipi: PhpBB
```

**Desteklenen URL Formatları:**
- `https://site.com/viewtopic.php?t=123`
- `https://site.com/viewforum.php?f=456`

#### Özel Siteler
```
Base URL: https://custom-site.com
Site Tipi: Custom
```

**Özel URL Listesi:**
```
https://custom-site.com/page1
https://custom-site.com/page2
https://custom-site.com/page3
```

### Spam Filtreleme Sistemi

#### Kara Liste Kelimeleri

**Basit Kelime Filtreleri:**
```
viagra
casino
porn
spam
pills
```

**Regex Filtreleri:**
```
/\b(viagra|casino|porn)\b/i
/\d{4}-\d{2}-\d{2}/  # Tarih formatı
/^[A-Z]{2,}$/        # Büyük harfli kelimeler
```

#### Beyaz Liste Kelimeleri

Sadece bu kelimeleri içeren içerikler çekilir:
```
tutorial
guide
help
documentation
```

#### HTML Element Filtreleri

Kaldırılacak HTML elementleri:
```
script
style
iframe
object
embed
```

### Anahtar Kelime Değiştirme

#### Basit Değiştirmeler

**Kelime Değiştirme:**
```
Eski Site → Yeni Site
Old Forum → New Forum
```

**URL Değiştirme:**
```
https://eski-site.com/ → https://yeni-site.com/
http://old-forum.com/ → http://new-forum.com/
```

#### Regex Değiştirmeleri

**E-posta Adresleri:**
```
Pattern: /\b(\w+)@eski-site\.com\b/
Replacement: $1@yeni-site.com
```

**Telefon Numaraları:**
```
Pattern: /(\d{3})-(\d{3})-(\d{4})/
Replacement: ($1) $2-$3
```

**Tarih Formatları:**
```
Pattern: /(\d{4})-(\d{2})-(\d{2})/
Replacement: $2/$3/$1
```

## 🔄 Otomatik Çalışma

### Cron Job Ayarları

**Varsayılan Çalışma Sıklığı:** Her 15 dakika

**Özel Zamanlama:**
```php
// Her 30 dakikada bir
$schedule = [
    'interval_minutes' => 30,
    'enabled' => 1
];

// Sadece iş saatlerinde (09:00-17:00)
$schedule = [
    'interval_minutes' => 60,
    'enabled' => 1,
    'time_restriction' => '09:00-17:00'
];
```

### Rate Limiting

**İstekler Arası Gecikme:**
- Minimum: 1 saniye
- Önerilen: 2-5 saniye
- Maksimum: 60 saniye

**Maksimum Post Sayısı:**
- Varsayılan: 10 post/çalışma
- Önerilen: 5-20 post/çalışma
- Maksimum: 100 post/çalışma

## 📊 İzleme ve Loglama

### Log Seviyeleri

#### Debug (Geliştirici)
```
[2024-01-15 10:30:45] DEBUG: İçerik çekiliyor: https://example.com/thread/123
[2024-01-15 10:30:46] DEBUG: Parser başlatıldı: XenforoParser
[2024-01-15 10:30:47] DEBUG: Başlık çıkarıldı: "Test Thread Title"
```

#### Info (Bilgi)
```
[2024-01-15 10:30:45] INFO: İçerik başarıyla çekildi: https://example.com/thread/123
[2024-01-15 10:30:46] INFO: Başarıyla yayınlanan içerik sayısı: 5
```

#### Warning (Uyarı)
```
[2024-01-15 10:30:45] WARNING: Hedef site erişilemiyor: https://example.com
[2024-01-15 10:30:46] WARNING: Spam içerik tespit edildi: https://example.com/thread/456
```

#### Error (Hata)
```
[2024-01-15 10:30:45] ERROR: HTTP hatası: Connection timeout
[2024-01-15 10:30:46] ERROR: Veritabanı hatası: Duplicate entry
```

### Log İstatistikleri

**Son 7 Gün:**
- Toplam Log: 1,234
- Hata: 12
- Uyarı: 45
- Bilgi: 1,177

**En Sık Hatalar:**
1. Bağlantı zaman aşımı (5)
2. Spam içerik (3)
3. Duplicate içerik (2)

### Log Temizleme

**Otomatik Temizleme:**
```php
// Son 30 günü temizle
$logger = new \AutoForumSyncPro\Service\Logger();
$logger->cleanLogs(30);
```

**Manuel Temizleme:**
1. Admin Panel → Loglar → Temizle
2. Gün sayısını seçin (1-365)
3. Temizle butonuna tıklayın

## 🛠️ Gelişmiş Kullanım

### Programatik Kullanım

#### Tek İçerik Çekme
```php
use AutoForumSyncPro\Service\ContentFetcher;
use AutoForumSyncPro\Service\ContentPublisher;

// Hedef siteyi ayarla
$targetSite = new \AutoForumSyncPro\Entity\TargetSite([
    'site_name' => 'Test Site',
    'base_url' => 'https://example.com',
    'site_type' => 'xenforo',
    'enabled' => 1,
    'default_user_id' => 1,
    'default_forum_id' => 2,
    'delay_between_requests' => 2
]);

// İçerik çek
$fetcher = new ContentFetcher();
$fetcher->setTargetSite($targetSite);
$content = $fetcher->fetchFromUrl('https://example.com/threads/test.123/');

if ($content) {
    // İçerik yayınla
    $publisher = new ContentPublisher();
    $result = $publisher->publish($content, [
        'user_id' => 1,
        'forum_id' => 2
    ]);
    
    if ($result) {
        echo "İçerik başarıyla yayınlandı!";
    }
}
```

#### Toplu İçerik Çekme
```php
$urls = [
    'https://example.com/threads/thread1.123/',
    'https://example.com/threads/thread2.124/',
    'https://example.com/threads/thread3.125/'
];

$contents = $fetcher->fetchMultiple($urls);

foreach ($contents as $content) {
    $publisher->publish($content);
}
```

#### Özel Parser Kullanma
```php
// Özel parser oluştur
class CustomParser extends \AutoForumSyncPro\Parser\AbstractParser
{
    public function parse($html, $url)
    {
        // Özel parsing mantığı
        return $content;
    }
    
    protected function extractTitle($html)
    {
        // Özel başlık çıkarma
        return $title;
    }
    
    // Diğer metodları implement et...
}

// Parser'ı kullan
$parser = new CustomParser();
$content = $parser->parse($html, $url);
```

### API Kullanımı

#### REST API Endpoints

**Hedef Site Ekleme:**
```bash
POST /api/auto-forum-sync/targets
Content-Type: application/json

{
    "site_name": "Test Site",
    "base_url": "https://example.com",
    "site_type": "xenforo",
    "default_user_id": 1,
    "default_forum_id": 2
}
```

**Manuel Senkronizasyon:**
```bash
POST /api/auto-forum-sync/sync
Content-Type: application/json

{
    "target_id": 1,
    "urls": [
        "https://example.com/threads/test.123/"
    ]
}
```

**Log Görüntüleme:**
```bash
GET /api/auto-forum-sync/logs?limit=50&level=error
```

### Webhook Desteği

**Webhook URL Ayarlama:**
```php
$webhookUrl = 'https://your-site.com/webhook/auto-forum-sync';
$webhookSecret = 'your-secret-key';

// Webhook gönder
$data = [
    'event' => 'content_published',
    'content_id' => 123,
    'thread_id' => 456,
    'timestamp' => time()
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $webhookUrl);
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'X-Webhook-Secret: ' . $webhookSecret
]);
curl_exec($ch);
curl_close($ch);
```

## 🔒 Güvenlik ve Etik

### Güvenlik Önerileri

#### 1. Rate Limiting
```php
// Agresif rate limiting
$targetSite->setDelayBetweenRequests(5); // 5 saniye
$targetSite->setMaxRequestsPerHour(100); // Saatte 100 istek
```

#### 2. User-Agent Rotasyonu
```php
$userAgents = [
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
    'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36',
    'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36'
];

$randomUserAgent = $userAgents[array_rand($userAgents)];
```

#### 3. Proxy Kullanımı
```php
$proxy = [
    'http' => 'http://proxy.example.com:8080',
    'https' => 'http://proxy.example.com:8080'
];

$client = new \GuzzleHttp\Client([
    'proxy' => $proxy,
    'timeout' => 30
]);
```

### Etik Kullanım

#### 1. Telif Hakkı Kontrolü
- Sadece kendi sitenizden içerik çekin
- İzin aldığınız sitelerden çekin
- Creative Commons lisanslı içerikleri tercih edin

#### 2. robots.txt Uyumu
```bash
# Hedef sitenin robots.txt dosyasını kontrol edin
curl https://example.com/robots.txt
```

#### 3. Kullanım Şartları
- Hedef sitenin kullanım şartlarını okuyun
- API kullanımı varsa API'yi tercih edin
- İletişime geçin ve izin alın

## 📈 Performans Optimizasyonu

### Veritabanı Optimizasyonu

#### İndeksler
```sql
-- Sık kullanılan sorgular için indeksler
CREATE INDEX idx_content_unique_id ON xf_auto_forum_sync_content(unique_id);
CREATE INDEX idx_content_status ON xf_auto_forum_sync_content(status);
CREATE INDEX idx_logs_created_date ON xf_auto_forum_sync_logs(created_date);
CREATE INDEX idx_logs_level ON xf_auto_forum_sync_logs(level);
```

#### Tablo Optimizasyonu
```sql
-- Tabloları optimize et
OPTIMIZE TABLE xf_auto_forum_sync_content;
OPTIMIZE TABLE xf_auto_forum_sync_logs;
```

### Bellek Optimizasyonu

#### PHP Ayarları
```ini
; php.ini
memory_limit = 512M
max_execution_time = 300
opcache.enable = 1
opcache.memory_consumption = 128
```

#### Garbage Collection
```php
// Bellek kullanımını optimize et
if (memory_get_usage() > 100 * 1024 * 1024) { // 100MB
    gc_collect_cycles();
}
```

### Ağ Optimizasyonu

#### HTTP/2 Desteği
```php
$client = new \GuzzleHttp\Client([
    'http_version' => '2.0',
    'timeout' => 30,
    'connect_timeout' => 10
]);
```

#### Keep-Alive Bağlantıları
```php
$client = new \GuzzleHttp\Client([
    'curl' => [
        CURLOPT_TCP_KEEPALIVE => 1,
        CURLOPT_TCP_KEEPIDLE => 10,
        CURLOPT_TCP_KEEPINTVL => 5
    ]
]);
```

## 🚨 Sorun Giderme

### Yaygın Sorunlar

#### 1. Bağlantı Hatası
```
ERROR: HTTP hatası: Connection timeout
```

**Çözüm:**
- Hedef site erişilebilir mi kontrol edin
- Firewall ayarlarını kontrol edin
- User-Agent string'ini değiştirin
- Proxy kullanın

#### 2. İçerik Çekilemiyor
```
ERROR: Başlık bulunamadı
```

**Çözüm:**
- Hedef sitenin HTML yapısı değişmiş olabilir
- Parser selector'larını güncelleyin
- JavaScript gerektiren içerikler çekilemez

#### 3. Spam Filtreleme Çok Sıkı
```
WARNING: Spam içerik tespit edildi
```

**Çözüm:**
- Beyaz liste kelimeleri ekleyin
- Kara liste kelimelerini gözden geçirin
- Regex pattern'lerini kontrol edin

#### 4. Duplicate İçerikler
```
WARNING: Duplicate içerik tespit edildi
```

**Çözüm:**
- Benzersiz ID çıkarma mantığını kontrol edin
- Veritabanındaki unique_id'leri kontrol edin
- URL formatlarını kontrol edin

### Debug Modu

#### Debug Modunu Etkinleştirme
```php
$logger = new \AutoForumSyncPro\Service\Logger();
$logger->setLogLevel('debug');
```

#### Debug Bilgileri
```php
// Detaylı debug bilgileri
$debugInfo = [
    'url' => $url,
    'html_length' => strlen($html),
    'title_found' => !empty($title),
    'content_found' => !empty($content),
    'unique_id' => $uniqueId,
    'processing_time' => microtime(true) - $startTime
];

$logger->debug('Debug bilgileri', $debugInfo);
```

## 📞 Destek ve Topluluk

### Destek Kanalları

1. **GitHub Issues**: Hata bildirimleri ve özellik istekleri
2. **Dokümantasyon**: Detaylı kullanım kılavuzu
3. **Topluluk Forumları**: XenForo topluluğu
4. **E-posta Desteği**: support@autoforumsync.com

### Katkıda Bulunma

1. **Fork** yapın
2. **Feature branch** oluşturun
3. **Commit** yapın
4. **Pull Request** gönderin

### Lisans

Bu eklenti MIT lisansı altında lisanslanmıştır.

---

**Not**: Bu kullanım kılavuzu sürekli güncellenmektedir. En güncel bilgiler için GitHub repository'sini kontrol edin.
