<?php
namespace XenBulletins\Core\Licensing;
use XenBulletins\Core\Licensing\StorageDriver\AbstractStorageDriver;

class Checker
{
    protected $endpoint;
    protected $addonId;
    protected $board_host;

    /**
     * @var \Closure
     */
    protected $remote_checker;

    /**
     * @var AbstractStorageDriver[]
     */
    protected $storageDrivers = array();

    public function __construct($storageDrivers)
    {
        /**
         * @var int $driverId
         * @var AbstractStorageDriver $storageDriver
         */
        foreach ($storageDrivers AS $driverId => $storageDriver)
        {
            if (
                ($storageDriver instanceof AbstractStorageDriver) === false
            )
            {
                unset($storageDrivers[$driverId]);
                continue;
            }
            $file = __FILE__;

            if ($storageDriver->isValid() === false)
            {
                unset($storageDrivers[$driverId]);
                continue;
            }
        }

        $this->storageDrivers = $storageDrivers;
    }

    public function getLicensePingUrl($licenseKey)
    {
        foreach ($this->storageDrivers AS $storageDriver)
        {
            $pingUrl = $storageDriver->getLicenseInfoUrl($licenseKey);
            if ($pingUrl !== false)
            {
                return $pingUrl;
            }
        }

        return false;
    }

    /**
     * @param $licenseKey
     * @return LicenseData
     * Returns updated license data from server
     */
    public function forceLicenseUpdate($licenseKey,$getNew=0)
    {
        // get the local version first
        $licenseData = $this->getLocalLicenseData($licenseKey);
    
        if ($licenseData === false or $getNew)
        {
            // no local data ever existed, create a new one
            $licenseData = new LicenseData();
        }

        $queryData = array(
            'license_key' => $licenseKey,
            'server_ip' => isset($_SERVER['SERVER_ADDR']) ? $_SERVER['SERVER_ADDR'] : '',
            'board_host' => $this->getBoardHost(),
            'addon_id'=> $this->getAddonId(),
            'ping_url' => $this->getLicensePingUrl($licenseKey)
        );
       
        if (
            empty($queryData['server_ip'])
            && function_exists('gethostbyname')
            && !empty($_SERVER['SERVER_NAME'])
        )
        {
            $queryData['server_ip'] = gethostbyname($_SERVER['SERVER_NAME']);
        }

        try
        {
            
           
            $licenseCheckString = $this->endpoint . '?' . http_build_query($queryData);
            
            $jsonResponse = call_user_func(
                $this->remote_checker,
                $licenseCheckString,
                $queryData,
                $licenseData
            );
            
        } catch (\Exception $exception)
        {
            $licenseData->setServerError($exception->getMessage());
            return $licenseData;
        }
        
        $licenseData->setLastServerResponse((array)$jsonResponse);
        $licenseData->setLastError('');

        if ($licenseData->isValid())
        {
            $licenseData->resetFailCount();
        }
        else
        {
            if ($licenseData->getLicenseErrorCode())
            {
                $licenseData->setLastError($licenseData->getLicenseErrorMessage() . ' (error code: ' . $licenseData->getLicenseErrorCode() . ')');
            }
        }

        return $licenseData;
    }

    public function setLicenseData($licenseKey, LicenseData $licenseData)
    {
        foreach ($this->storageDrivers AS $storageDriver)
        {
            
            
            $storageDriver->setLocalData($licenseKey, $licenseData);
        }
    }

    public function getLocalLicenseData($licenseKey)
    {
        
        if (!$licenseKey)
        {
            return false;
        }
        
        /** @var AbstractStorageDriver $storageDriver */
        foreach ($this->storageDrivers AS $storageDriver)
        {
            $licenseData = $storageDriver->getLocalData($licenseKey);
            
            if (
                $licenseData !== false
                && $licenseData->checkDataIntegrity()
            )
            {
                
                return $licenseData;
            }
        }

        return false;
    }

    /**
     * @return string
     */
    public function getEndpoint()
    {
        return $this->endpoint;
    }

    
     public function getAddonId()
    {
        return $this->addonId;
    }
    
    
    
    /**
     * @param string $endpoint
     */
    public function setEndpoint($endpoint)
    {
        $this->endpoint = $endpoint;

        return $this;
    }

     public function setAddonId($addonId)
    {
        $this->addonId = $addonId;

        return $this;
    }
    
    
    
    
    public function getBoardHost()
    {
        return $this->board_host;
    }

    public function setBoardHost($board_host)
    {
        $this->board_host = $board_host;
    }

    public function getRemoteChecker()
    {
        return $this->remote_checker;
    }

    public function setRemoteChecker($remote_checker)
    {
        $this->remote_checker = $remote_checker;
    }
}
