<?php
/**
 * Google Indexing API Script
 * Simple script to submit URLs to Google for indexing
 * Supports sitemap parsing and cron job execution
 */

// Configuration
$config = [
    'service_account_file' => 'service-account-key.json', // Google Service Account JSON file
    'sitemap_url' => 'https://yoursite.com/sitemap.xml', // Your sitemap URL
    'batch_size' => 10, // Number of URLs to process per batch
    'delay_between_batches' => 2, // Seconds to wait between batches
    'log_file' => 'indexer.log'
];

// Logging function
function logMessage($message) {
    global $config;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($config['log_file'], $logEntry, FILE_APPEND | LOCK_EX);
    echo $logEntry;
}

// Check if running from command line (cron) or web
$isCli = php_sapi_name() === 'cli';

if (!$isCli) {
    // Web interface
    ?>
    <!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Google Indexer</title>
        <style>
            body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; background: #f8f9fa; }
            .container { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .header { text-align: center; margin-bottom: 30px; }
            .header h1 { color: #4285f4; margin: 0; }
            .form-group { margin-bottom: 20px; }
            label { display: block; margin-bottom: 8px; font-weight: bold; color: #333; }
            input, textarea { width: 100%; padding: 12px; border: 2px solid #e1e5e9; border-radius: 6px; font-size: 14px; box-sizing: border-box; }
            input:focus, textarea:focus { outline: none; border-color: #4285f4; }
            button { background: #4285f4; color: white; padding: 12px 24px; border: none; border-radius: 6px; cursor: pointer; font-size: 16px; margin-right: 10px; }
            button:hover { background: #3367d6; }
            .log { background: #1a1a1a; color: #00ff00; padding: 20px; border-radius: 6px; font-family: 'Courier New', monospace; max-height: 400px; overflow-y: auto; white-space: pre-wrap; }
            .status { padding: 15px; margin: 15px 0; border-radius: 6px; }
            .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
            .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
            .info { background: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>🚀 Google Indexer</h1>
                <p>URL'lerinizi Google'a hızlıca gönderin</p>
            </div>
            
            <form method="post">
                <div class="form-group">
                    <label for="sitemap_url">📍 Sitemap URL:</label>
                    <input type="url" id="sitemap_url" name="sitemap_url" value="<?php echo htmlspecialchars($config['sitemap_url']); ?>" placeholder="https://yoursite.com/sitemap.xml">
                </div>
                
                <div class="form-group">
                    <label for="urls">📝 Manuel URL'ler (her satıra bir URL):</label>
                    <textarea id="urls" name="urls" rows="5" placeholder="https://example.com/page1&#10;https://example.com/page2&#10;https://example.com/page3"></textarea>
                </div>
                
                <button type="submit" name="action" value="index">📤 URL'leri İndexle</button>
                <button type="submit" name="action" value="sitemap">🗺️ Sitemap'ten İndexle</button>
            </form>
            
            <?php
            if ($_POST) {
                echo '<div class="log">';
                echo '<h3>📊 İşlem Logları:</h3>';
                
                if ($_POST['action'] === 'index' && !empty($_POST['urls'])) {
                    $urls = array_filter(array_map('trim', explode("\n", $_POST['urls'])));
                    processUrls($urls);
                } elseif ($_POST['action'] === 'sitemap') {
                    $sitemapUrl = $_POST['sitemap_url'] ?? $config['sitemap_url'];
                    $urls = parseSitemap($sitemapUrl);
                    if ($urls) {
                        processUrls($urls);
                    }
                }
                echo '</div>';
            }
            ?>
            
            <div class="status info">
                <strong>💡 Kullanım:</strong><br>
                • Manuel URL'ler: İndexlemek istediğiniz URL'leri her satıra bir tane yazın<br>
                • Sitemap: Sitemap URL'nizi girin ve tüm URL'leri otomatik olarak işleyin<br>
                • Cron Job: <code>php indexer.php</code> komutu ile otomatik çalıştırın
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Command line execution
if ($isCli) {
    logMessage("Google Indexer başlatıldı (Cron)");
    
    // Parse sitemap and submit URLs
    $urls = parseSitemap($config['sitemap_url']);
    if ($urls) {
        processUrls($urls);
    } else {
        logMessage("HATA: Sitemap parse edilemedi");
    }
}

/**
 * Parse sitemap and extract URLs
 */
function parseSitemap($sitemapUrl) {
    logMessage("Sitemap parse ediliyor: $sitemapUrl");
    
    $context = stream_context_create([
        'http' => [
            'timeout' => 30,
            'user_agent' => 'Google-Indexer/1.0'
        ]
    ]);
    
    $sitemapContent = @file_get_contents($sitemapUrl, false, $context);
    
    if ($sitemapContent === false) {
        logMessage("HATA: Sitemap indirilemedi");
        return false;
    }
    
    $xml = @simplexml_load_string($sitemapContent);
    if ($xml === false) {
        logMessage("HATA: Sitemap XML parse edilemedi");
        return false;
    }
    
    $urls = [];
    
    // Handle sitemap index (contains multiple sitemaps)
    if (isset($xml->sitemap)) {
        foreach ($xml->sitemap as $sitemap) {
            $subSitemapUrl = (string)$sitemap->loc;
            logMessage("Alt sitemap bulundu: $subSitemapUrl");
            $subUrls = parseSitemap($subSitemapUrl);
            if ($subUrls) {
                $urls = array_merge($urls, $subUrls);
            }
        }
    }
    
    // Handle regular sitemap (contains URLs)
    if (isset($xml->url)) {
        foreach ($xml->url as $url) {
            $urls[] = (string)$url->loc;
        }
    }
    
    logMessage("Toplam " . count($urls) . " URL bulundu");
    return $urls;
}

/**
 * Process URLs and submit to Google Indexing API
 */
function processUrls($urls) {
    global $config;
    
    if (empty($urls)) {
        logMessage("İşlenecek URL bulunamadı");
        return;
    }
    
    logMessage("Toplam " . count($urls) . " URL işlenecek");
    
    // Initialize Google API client
    $client = initializeGoogleClient();
    if (!$client) {
        logMessage("HATA: Google API client başlatılamadı");
        return;
    }
    
    $indexingService = new Google_Service_Indexing($client);
    
    // Process URLs in batches
    $batches = array_chunk($urls, $config['batch_size']);
    $totalProcessed = 0;
    $totalSuccess = 0;
    $totalErrors = 0;
    
    foreach ($batches as $batchIndex => $batch) {
        logMessage("Batch " . ($batchIndex + 1) . "/" . count($batches) . " işleniyor (" . count($batch) . " URL)");
        
        foreach ($batch as $url) {
            $totalProcessed++;
            
            try {
                $urlNotification = new Google_Service_Indexing_UrlNotification();
                $urlNotification->setUrl($url);
                $urlNotification->setType('URL_UPDATED');
                
                $result = $indexingService->urlNotifications->publish($urlNotification);
                
                if ($result->getUrlNotificationMetadata()) {
                    $totalSuccess++;
                    logMessage("✓ Başarılı: $url");
                } else {
                    $totalErrors++;
                    logMessage("✗ Hata: $url - Metadata alınamadı");
                }
                
            } catch (Exception $e) {
                $totalErrors++;
                logMessage("✗ Hata: $url - " . $e->getMessage());
            }
            
            // Small delay between requests
            usleep(100000); // 0.1 second
        }
        
        // Delay between batches
        if ($batchIndex < count($batches) - 1) {
            logMessage("Sonraki batch için " . $config['delay_between_batches'] . " saniye bekleniyor...");
            sleep($config['delay_between_batches']);
        }
    }
    
    logMessage("İşlem tamamlandı. Toplam: $totalProcessed, Başarılı: $totalSuccess, Hata: $totalErrors");
}

/**
 * Initialize Google API client
 */
function initializeGoogleClient() {
    global $config;
    
    if (!file_exists($config['service_account_file'])) {
        logMessage("HATA: Service account dosyası bulunamadı: " . $config['service_account_file']);
        return false;
    }
    
    try {
        $client = new Google_Client();
        $client->setAuthConfig($config['service_account_file']);
        $client->addScope('https://www.googleapis.com/auth/indexing');
        $client->setUseBatch(true);
        
        return $client;
    } catch (Exception $e) {
        logMessage("HATA: Google Client başlatılamadı: " . $e->getMessage());
        return false;
    }
}

// Include Google API library (you need to install it via Composer)
// require_once 'vendor/autoload.php';

// If Google API library is not available, show installation instructions
if (!class_exists('Google_Client')) {
    if ($isCli) {
        logMessage("HATA: Google API library bulunamadı. Composer ile yükleyin: composer require google/apiclient");
    } else {
        echo '<div class="status error">Google API library bulunamadı. Composer ile yükleyin: <code>composer require google/apiclient</code></div>';
    }
    exit;
}
?>
