﻿<?php

namespace Ghf\TradeFeedback\Service;

use Ghf\TradeFeedback\Entity\Trade;
use XF\Entity\User;
use XF\PrintableException;
use XF\Service\AbstractService;
use XF\Util\Ip;

class TradeCreator extends AbstractService
{
    /** @var Trade */
    protected $trade;

    /** @var User|null */
    protected $sender;

    /** @var User|null */
    protected $receiver;

    public function __construct(\XF\App $app)
    {
        parent::__construct($app);
        $this->trade = $this->em()->create('Ghf\TradeFeedback:Trade');
    }

    public function getTrade(): Trade
    {
        return $this->trade;
    }

    public function setSender(User $sender): void
    {
        $this->sender = $sender;
        $this->trade->sender_id = $sender->user_id;
    }

    public function setReceiver(User $receiver): void
    {
        $this->receiver = $receiver;
        $this->trade->receiver_id = $receiver->user_id;
    }

    public function setRating(string $rating): void
    {
        $this->trade->rating = $rating;
    }

    public function setMessage(string $message): void
    {
        $this->trade->message = trim($message);
    }

    public function save(): Trade
    {
        $this->validate();

        $this->trade->trade_date = \XF::$time;
        $ip = $this->app->request()->getIp();
        $this->trade->ip_address = $ip ? Ip::convertIpStringToBinary($ip) : null;

        $this->trade->preSave();
        if ($errors = $this->trade->getErrors())
        {
            throw new PrintableException(reset($errors));
        }

        $this->trade->save();
        return $this->trade;
    }

    protected function validate(): void
    {
        if (!$this->sender || !$this->receiver)
        {
            throw new PrintableException('Sender or receiver missing.');
        }

        if ($this->sender->user_id === $this->receiver->user_id)
        {
            throw new PrintableException('You cannot leave feedback for yourself.');
        }

        $options = $this->options();
        $message = $this->trade->message;

        if ($options->requireMessage && $message === '')
        {
            throw new PrintableException('A feedback message is required.');
        }

        if ($options->maxDailyFeedbackPerUser)
        {
            /** @var \Ghf\TradeFeedback\Repository\Trade $tradeRepo */
            $tradeRepo = $this->repository('Ghf\TradeFeedback:Trade');
            $count = $tradeRepo->countFeedbackGivenBySenderToday(
                $this->sender->user_id,
                $this->receiver->user_id
            );

            if ($count >= $options->maxDailyFeedbackPerUser)
            {
                throw new PrintableException('You have reached the daily feedback limit for this user.');
            }
        }

        if ($options->negativeNeedsApproval && $this->trade->rating === 'negative')
        {
            throw new PrintableException('Negative feedback currently requires moderator approval.');
        }

        $allowed = ['positive', 'negative', 'neutral'];
        if (!in_array($this->trade->rating, $allowed, true))
        {
            throw new PrintableException('Invalid rating selected.');
        }
    }
}

